/* ============ */
/* bitmanip.c	*/
/* ============ */
/* -------------------------------------------------------------------- *
 *									*
 *		    King Baker Enterprises, Inc.			*
 *		       Stillwater, Oklahoma				*
 *									*
 *			 MODULE PROLOGUE				*
 *									*
 *  Name:	bitmanip.c						*
 *									*
 *  Date:	24 October 1992						*
 *									*
 *  Version:	1.0							*
 *									*
 *  Purpose:	Provide bit manipulation services.			*
 *									*
 *  Design Document:	None.						*
 *									*
 *  Author:  K. B. Williams						*
 *									*
 *  		       MODIFICATION HISTORY				*
 *									*
 *  Name of Modifier	Date/Time	Description			*
 *  ----------------	---------	-----------			*
 *									*
 * -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- */
/*		  BIPMANIP - BIT MANIPULATION FUNCTIONS 		*/
/* -------------------------------------------------------------------- */

#include "bit_hlrs.h"

static
UCHAR	bit_mask[] = {	1 << 7, 1 << 6, 1 << 5, 1 << 4,
			1 << 3, 1 << 2, 1 << 1, 1
		     };
/* INDENT ON */

/* ====================================================================	*/
/* ClrBitArray - sets num_bits in bit_ary at bit_posn to bit_val	*/
/* ====================================================================	*/
/* INDENT OFF */
# if defined(__STDC__) || defined(__PROTO__)
void
ClrBitArray(UCHAR *bit_ary, UINT bit_posn, UINT num_bits, UCHAR bit_val)
# else
void
ClrBitArray(bit_ary, bit_posn, num_bits, bit_val)
UCHAR	*bit_ary;
UINT	 bit_posn;
UINT	 num_bits;
UCHAR	 bit_val;
# endif
/* INDENT ON */
{
     UINT    i;

    for (i = 0; i < num_bits; ++i)
    {
	SetBit(bit_ary, bit_posn + i, bit_val);
    }
}
/* ==================================================================== */
/* CmplBit - complements bit at bit_posn in bit_ary			*/
/* ==================================================================== */
/* INDENT OFF */
# if defined(__STDC__) || defined(__PROTO__)
void
CmplBit(UCHAR *bit_ary, UINT bit_posn)
# else
void
CmplBit(bit_ary, bit_posn)
UCHAR	*bit_ary;
UINT	 bit_posn;
# endif
/* INDENT ON */
{
    SetBit(bit_ary, bit_posn, (UCHAR)!TestBit(bit_ary, bit_posn));
}
/* ======================================================================== */
/* CopyBits - copies bits in srce_bits @ srce_posn to dest_bits @ dest_posn */
/* ======================================================================== */
/* INDENT OFF */
# if defined(__STDC__) || defined(__PROTO__)
void
CopyBits(UCHAR *srce_bits, UINT srce_posn, UINT num_bits,
	UCHAR *dest_bits, UINT dest_posn)
# else
void
CopyBits(srce_bits, srce_posn, num_bits, dest_bits, dest_posn)
UCHAR	*srce_bits;
UINT	 srce_posn;
UINT	 num_bits;
UCHAR	*dest_bits;
UINT	 dest_posn;
# endif
/* INDENT ON */
{
    UINT    i;
    UCHAR   *srce_bit_addr, *dest_bit_addr;

    srce_bit_addr = srce_bits + (srce_posn >> 3) + (srce_posn & 0x07);
    dest_bit_addr = dest_bits + (dest_posn >> 3) + (dest_posn & 0x07);

    if (srce_bit_addr >= dest_bit_addr)	/* copy bits forward */
    {
	for (i = 0; i < num_bits; ++i)
	{
	    SetBit(dest_bits, dest_posn + i,
		TestBit(srce_bits, srce_posn + i));
	}
    }
    else				/* copy bits backward */
    {
	for (i = num_bits; i > 0; --i)
	{
	    SetBit(dest_bits, dest_posn + i - 1,
		TestBit(srce_bits, srce_posn + i - 1));
	}
    }
}
/* ====================================================================	*/
/* ReverseBits - copies num_bits backwards from srce_ary to dest_ary	*/
/* ====================================================================	*/
/* INDENT OFF */
# if defined(__STDC__) || defined(__PROTO__)
void
ReverseBits(UCHAR *srce_ary, UINT srce_posn, UINT num_bits,
	     UCHAR *dest_ary, UINT dest_posn)
# else
void
ReverseBits(srce_ary, srce_posn, num_bits, dest_ary, dest_posn)
UCHAR	*srce_ary;
UINT	 srce_posn;			/* starting bit posn in srce_ary */
UINT	 num_bits;
UCHAR	*dest_ary;
UINT	 dest_posn;			/* starting bit posn in dest_ary */
# endif
/* INDENT ON */
{
    UINT    i;

    if (srce_ary == dest_ary)
    {
	ReverseInSitu(srce_ary, srce_posn, num_bits);
    }
    else
    {
	for (i = 0; i < num_bits; ++i)
	{
	    SetBit(dest_ary, dest_posn + i,
		TestBit(srce_ary, (srce_posn + num_bits - 1) - i));
	}
    }
}
/* ====================================================================	*/
/* ReverseInSitu - reverses order of bits in srce_ary in situ		*/
/* ====================================================================	*/
/* INDENT OFF */
# if defined(__STDC__) || defined(__PROTO__)
void
ReverseInSitu(UCHAR *srce_ary, UINT srce_posn, UINT num_bits)
# else
void
ReverseInSitu(srce_ary, srce_posn, num_bits)
UCHAR	*srce_ary;
UINT	 srce_posn;
UINT	 num_bits;
# endif
/* INDENT ON */
{
    UINT    i;

    for (i = 0; i < num_bits >> 1; ++i)
    {
	UCHAR	temp = TestBit(srce_ary, srce_posn + i);

	SetBit(srce_ary, srce_posn + i,
	    TestBit(srce_ary, srce_posn + (num_bits - 1) - i));

	SetBit(srce_ary, srce_posn + (num_bits - 1) - i, temp);
    }
}
/* ==================================================================== */
/* SetBit - sets bit at bit_posn in bit_ary to bit_val 		*/
/* ==================================================================== */
/* INDENT OFF */
# if defined(__STDC__) || defined(__PROTO__)
void
SetBit(UCHAR *bit_ary, UINT bit_posn, UCHAR bit_val)
# else
void
SetBit(bit_ary, bit_posn, bit_val)
UCHAR	*bit_ary;
UINT	 bit_posn;
UCHAR	 bit_val;
# endif
/* INDENT ON */
    {
    if (bit_val)			/* set the bit */
    {
	bit_ary[bit_posn >> 3] |= bit_mask[bit_posn & 0x07];
    }
    else				/* clear the bit */
    {
	bit_ary[bit_posn >> 3] &= ~bit_mask[bit_posn & 0x07];
    }
}
/* ==================================================================== */
/* TestBit = returns 1 if bit at bit_posn in bit_ary is a 1, else 0	*/
/* ==================================================================== */
/* INDENT OFF */
# if defined(__STDC__) || defined(__PROTO__)
UCHAR
TestBit(UCHAR *bit_ary, UINT bit_posn)
# else
UCHAR
TestBit(bit_ary, bit_posn)
UCHAR	*bit_ary;
UINT	 bit_posn;
# endif
/* INDENT ON */
{
    return((UCHAR)(bit_ary[bit_posn >> 3] & bit_mask[bit_posn & 0x07]));
}
# ifdef DEBUG
#include <stdio.h>
/* ------------------- */
/* FUNCTION PROTOTYPES */
/* ------------------- */
# undef F
# if defined(__STDC__) || defined(__PROTO__)
#     define  F( P )  P
# else
#     define  F( P )  ()
# endif

/* INDENT OFF */
extern	void	main F(( void ));
extern	void	show_map F(( UCHAR * ));
extern	void	test_manip F(( void ));

# undef F
/* INDENT ON */

void
main()
{
	test_manip();
}
void
test_manip()
{
	int	i;

	UINT	bit_num, from, num_bits, to;
	UCHAR	set;
	static
	UCHAR	bit_map[8] = { 0xAB, 0xCD, 0xEF, 'a', 'b', 'c', '1', '2'};
	static
	UCHAR	new_map[8];

    puts("Starting new_map");
    show_map(new_map);

    ClrBitArray(new_map, 0, 64, 1);

    puts("All bits set to 1:");
    show_map(new_map);

    ClrBitArray(new_map, 0, 64, 0);

    puts("All bits set to 0:");
    show_map(new_map);

    for (i = 1; i < 64; i+=2)
    {
	CmplBit(new_map, i);
    }

    puts("Odd-numbered bits complemented to 1");
    show_map(new_map);

    for (i = 1; i < 64; i+=2)
    {
	CmplBit(new_map, i);
    }

    puts("Odd-numbered bits recomplemented to 0");
    show_map(new_map);

    ClrBitArray(new_map, 0, 1, 1);

    puts("First bit set to 1:");
    show_map(new_map);

    ClrBitArray(new_map, 63, 1, 1);

    puts("Last bit set to 1:");
    show_map(new_map);

    ClrBitArray(new_map, 7, 2, 1);

    puts("Two bits set to 1 at position 7:");
    show_map(new_map);

    puts("Starting bit_map");
    show_map(bit_map);

    puts("Reverse bits in situ:");
    printf("from %x", bit_map[0]);
    ReverseInSitu(bit_map, 0, 64);

    printf(" to %x\n", bit_map[7]);
    show_map(bit_map);

    while (main)
    {
	ClrBitArray(new_map, 0, 64, 0);

	printf("\nBit number: ");
	scanf("%d", &bit_num);
	printf("1 to set, 0 to clear: ");
	scanf("%d", &set);

	SetBit(bit_map, bit_num, set);

	puts("New map");
	show_map(bit_map);

	printf("Extract from bit #: ");
	scanf("%d", &from);
	printf("Number of bits: ");
	scanf("%d", &num_bits);

	printf("\nCopy into position: ");
	scanf("%d", &to);

	CopyBits(bit_map, from, num_bits, new_map, to);

	puts("Extracted map (NEW BIT ARRAY)");
	show_map(new_map);
    }
}
void
show_map(UCHAR *bit_map)
{
    UINT    i;

    puts("0    v    1    v    2    v    3    v    4    v    5    v    6");
    puts(".....|....0....|....0....|....0....|....0....|....0....|....0...");
    for (i = 0; i < 64; ++i)
    {
	putchar(TestBit(bit_map, i) ? 'X' : '.');
    }
    printf("\t\tpress <CR>");
    getchar();
}
# endif
